
#ifndef BASIC_DATA_H
#define BASIC_DATA_H

#include <iostream>
#include <stdio.h>
#include <math.h>
#include <time.h>

#define cuFDType float

struct cuComplex {
    cuFDType   r;
    cuFDType   i;
    cuComplex( cuFDType a, cuFDType b ) : r(a), i(b)  {}
    cuComplex() : r(0), i(0)  {}
    cuComplex( cuFDType a) : r(a), i(0)  {}
    __device__ __host__ cuFDType magnitude2( void ) {
        return r * r + i * i;
    }
    __device__ __host__ cuFDType abs( void ) const {
        return sqrtf(r * r + i * i);
    }

    __device__ __host__ cuComplex operator*(const cuComplex& a) const{
        return cuComplex(r*a.r - i*a.i, i*a.r + r*a.i);
    }
    __device__ __host__ cuComplex operator*(const cuFDType& a) const{
        return cuComplex(r*a , i*a);
    }
    __device__ __host__ cuComplex operator+(const cuComplex& a) const {
        return cuComplex(r+a.r, i+a.i);
    }
    __device__ __host__ cuComplex operator-(const cuComplex& a) const {
        return cuComplex(r-a.r, i-a.i);
    }
    __device__ __host__ cuComplex operator-() const {
        return cuComplex(-r, -i);
    }

    __device__ __host__ void operator+=(const cuComplex& a) {
        r+=a.r;
	i+=a.i;
    }
    __device__ __host__ void operator-=(const cuComplex& a) {
        r-=a.r;
	i-=a.i;
    }

    __device__ __host__ cuComplex conj() const{
	return cuComplex(r,-i);
    }

    __device__ __host__ cuComplex sign() const {
      cuFDType a = abs();
      if (a==0.0f)
        return 0;
      else
        return cuComplex(r/a,i/a);
    }

    void print(void){
      printf("# %f + %fi \n ",r, i);
    }
};




struct cu2dMatrix;


struct cu2dVect {
    cuComplex   value1;
    cuComplex   value2;

    cu2dVect( cuComplex a, cuComplex b) : value1(a), value2(b) {}
    cu2dVect() : value1(0.0), value2(0.0) {}

    __device__ __host__ cu2dVect operator+(const cu2dVect& a) const {
      return cu2dVect(value1+a.value1,value2+a.value2);
    }
    __device__ __host__ cu2dVect nonLinearF() const {
        return cu2dVect(value1.sign(),value2.sign());
    }
    __device__ __host__ cu2dVect conj() const {
        return cu2dVect(value1.conj(),value2.conj());
    }

    __device__ __host__ cu2dMatrix kProd(const cu2dVect a) const ;

    void print(void){
      //printf("%f \n%f\n\n",value1, value2);

      value1.print();
      value2.print();
    }
};


struct cu2dMatrix {
    cuComplex   value11;
    cuComplex   value12;
    cuComplex   value21;
    cuComplex   value22;

    cu2dMatrix() : value11(0.0), value12(0.0), value21(0.0), value22(0.0)  {}
    cu2dMatrix( cuComplex a, cuComplex b, cuComplex c, cuComplex d ) : value11(a), value12(b), value21(c), value22(d)  {}


    __device__ __host__ cu2dMatrix operator*(const cu2dMatrix& a) const {
      return cu2dMatrix(value11*a.value11 + value21*a.value12,
                        value12*a.value11 + value22*a.value12,
                        value11*a.value21 + value21*a.value22,
                        value12*a.value21 + value22*a.value22);
    }

    __device__ __host__ cu2dVect operator*(const cu2dVect& a) const {
      return cu2dVect(value11*a.value1 + value21*a.value2,
		      value12*a.value1 + value22*a.value2);
    }

    __device__ __host__ cu2dMatrix operator*(const cuComplex& a) const {
      return cu2dMatrix(value11*a, value12*a, value21*a, value22*a);
    }


    __device__ __host__ cu2dMatrix operator+(const cu2dMatrix& a) const {
      return cu2dMatrix(value11+a.value11,value12+a.value12,value21+a.value21,value22+a.value22);
    }
    __device__ __host__ cu2dMatrix operator-(const cu2dMatrix& a) const {
      return cu2dMatrix(value11-a.value11,value12-a.value12,value21-a.value21,value22-a.value22);
    }

    __device__ __host__ void operator+=(const cu2dMatrix& a) {
      value11+=a.value11;
      value12+=a.value12;
      value21+=a.value21;
      value22+=a.value22;
    }
    __device__ __host__ void operator-=(const cu2dMatrix& a) {
      value11-=a.value11;
      value12-=a.value12;
      value21-=a.value21;
      value22-=a.value22;
    }

    void print(void){
        printf("[ %f + %fi, %f + %fi;  %f + %fi, %f + %fi] \n",
        value11.r, value11.i,
        value21.r, value21.i,
        value12.r, value12.i,
        value22.r, value22.i);
    }
};





/*
 *
 * 3d - datatypes
 *
 */



struct cu3dMatrix;


struct cu3dVect {
    cuComplex   value1;
    cuComplex   value2;
    cuComplex   value3;

    cu3dVect( cuComplex a, cuComplex b, cuComplex c) : value1(a), value2(b), value3(c) {}
    cu3dVect() : value1(0.0), value2(0.0), value3(0.0) {}

    __device__ __host__ cu3dVect operator+(const cu3dVect& a) const {
      return cu3dVect(value1+a.value1,value2+a.value2,value3+a.value3);
    }
    __device__ __host__ cu3dVect nonLinearF() const {
        return cu3dVect(value1.sign(),value2.sign(),value3.sign());
    }
    __device__ __host__ cu3dVect conj() const {
        return cu3dVect(value1.conj(),value2.conj(),value3.conj());
    }

    __device__ __host__ cu3dMatrix kProd(const cu3dVect a) const ;

    void print(void){
      //printf("%f \n%f\n\n",value1, value2);

      value1.print();
      value2.print();
      value3.print();
    }
};







struct cu3dMatrix {
    cuComplex   value11;
    cuComplex   value12;
    cuComplex   value13;
    cuComplex   value21;
    cuComplex   value22;
    cuComplex   value23;
    cuComplex   value31;
    cuComplex   value32;
    cuComplex   value33;

    __device__ __host__ cu3dMatrix() : value11(0.0), value12(0.0), value13(0.0), value21(0.0), value22(0.0), value23(0.0), value31(0.0), value32(0.0), value33(0.0)  {}
    cu3dMatrix( cuComplex a, cuComplex b, cuComplex c, cuComplex d, cuComplex e, cuComplex f, cuComplex g, cuComplex h, cuComplex i ) : value11(a), value12(b), value13(c), value21(d), value22(e), value23(f), value31(g), value32(h), value33(i)  {}


    __device__ __host__ cu3dMatrix operator*(const cu3dMatrix& a) const {
      return cu3dMatrix(value11*a.value11 + value21*a.value12 + value31*a.value13 ,
                        value12*a.value11 + value22*a.value12 + value32*a.value13 ,
                        value13*a.value11 + value23*a.value12 + value33*a.value13 ,

                        value11*a.value21 + value21*a.value22 + value31*a.value23,
                        value12*a.value21 + value22*a.value22 + value32*a.value23,
                        value13*a.value21 + value23*a.value22 + value33*a.value23,

                        value11*a.value31 + value21*a.value32 + value31*a.value33,
                        value12*a.value31 + value22*a.value32 + value32*a.value33,
                        value13*a.value31 + value23*a.value32 + value33*a.value33);
    }

    __device__ __host__ cu3dVect operator*(const cu3dVect& a) const {
      return cu3dVect(value11*a.value1 + value21*a.value2 + value31*a.value3,
                      value12*a.value1 + value22*a.value2 + value32*a.value3,
                      value13*a.value1 + value23*a.value2 + value33*a.value3);
    }

    __device__ __host__ cu3dMatrix operator*(const cuFDType& a) const {
      return cu3dMatrix(value11*a, value12*a, value13*a,
                        value21*a, value22*a, value23*a,
                        value31*a, value32*a, value33*a);
    }


    __device__ __host__ cu3dMatrix operator*(const cuComplex& a) const {
      return cu3dMatrix(value11*a, value12*a, value13*a,
                        value21*a, value22*a, value23*a,
                        value31*a, value32*a, value33*a);
    }

    __device__ __host__ cu3dMatrix operator+(const cu3dMatrix& a) const {
      return cu3dMatrix(value11+a.value11,value12+a.value12,value13+a.value13,
                        value21+a.value21,value22+a.value22,value23+a.value23,
                        value31+a.value31,value32+a.value32,value33+a.value33);
    }

    __device__ __host__ cu3dMatrix operator-(const cu3dMatrix& a) const {
      return cu3dMatrix(value11-a.value11,value12-a.value12,value13-a.value13,
                        value21-a.value21,value22-a.value22,value23-a.value23,
                        value31-a.value31,value32-a.value32,value33-a.value33);
    }

    __device__ __host__ void operator+=(const cu3dMatrix& a) {
      value11+=a.value11;
      value12+=a.value12;
      value13+=a.value13;
      value21+=a.value21;
      value22+=a.value22;
      value23+=a.value23;
      value31+=a.value31;
      value32+=a.value32;
      value33+=a.value33;
    }
    __device__ __host__ void operator-=(const cu3dMatrix& a) {
      value11-=a.value11;
      value12-=a.value12;
      value13-=a.value13;
      value21-=a.value21;
      value22-=a.value22;
      value23-=a.value23;
      value31-=a.value31;
      value32-=a.value32;
      value33-=a.value33;
    }

    void print(void){
        printf("[ %f + %fi, %f + %fi, %f + %fi, %f + %fi;  %f + %fi, %f + %fi] \n",
        value11.r, value11.i,
        value21.r, value21.i,
        value31.r, value31.i,
        value12.r, value12.i,
        value22.r, value22.i,
        value23.r, value23.i);
    }
};









#endif // BASIC_DATA_H

